# AI Migration Assistant Instructions

This file contains instructions for AI assistants to help migrate Abyss UI components from V1 to V2.

## Migration Context

The migration data is stored in the `MigrationData.json` file, which contains comprehensive migration information for all components. Each component entry in the JSON includes:

- `v1Import` and `v2Import` objects with import paths
- `overview` describing the key changes
- Component-specific change arrays (e.g., `buttonChanges`, `accordionChanges`)

## Assistance Tasks

When asked about migrating a component, the AI should:

1. Identify the component being migrated
2. Reference the appropriate migration data from `MigrationData.json`
3. Provide step-by-step migration instructions based on the component's change arrays
4. Offer to transform code snippets directly

## Component Migration Patterns

### Imports

- ALL V2 components must follow this import pattern: `import { V2ComponentName as ComponentName } from '@uhg-abyss/web/ui/ComponentName';`
- This applies to ALL components being migrated to V2, even if the component structure/name doesn't change
- Examples:
  - Button: `import { V2Button as Button } from '@uhg-abyss/web/ui/Button';`
  - Accordion: `import { V2Accordion as Accordion } from '@uhg-abyss/web/ui/Accordion';`
  - SelectInput: `import { V2SelectInput as SelectInput } from '@uhg-abyss/web/ui/SelectInput';`
- This pattern preserves the component name in the JSX while using the V2 implementation

### Props

- Identify renamed props
- Find replacements for removed props
- Explain new prop patterns

### Component Structure

- Guide through structural changes in components
- Explain composition pattern changes

## Example Responses

When asked "How do I migrate Button component?":

````
To migrate the Button component from V1 to V2:

1. Import changes: Use V2 prefix with alias
   ```jsx
   // V1 import
   import { Button } from '@uhg-abyss/web/ui/Button';

   // V2 import
   import { V2Button as Button } from '@uhg-abyss/web/ui/Button';
````

2. Update variant and color props:

   - V1 variants are split into variant and color in V2:
     - 'solid' → variant: 'filled', color: 'brand'
     - 'outline' → variant: 'outline', color: 'brand'
     - 'tertiary' → variant: 'text', color: 'brand'
     - 'destructive' → variant: 'filled', color: 'destructive'
     - 'alternate' → variant: 'filled', color: 'inverse'
     - 'ghost' → variant: 'text', color: 'neutral'

3. Update icon handling:

   - Replace before/after props with unified icon prop:

     ```jsx
     // V1
     <Button before={<IconSymbol icon="home" />}>Home</Button>
     <Button after={<IconSymbol icon="arrow_forward" />}>Next</Button>

     // V2
     <Button icon={{ icon: "home", position: "leading" }}>Home</Button>
     <Button icon={{ icon: "arrow_forward", position: "trailing" }}>Next</Button>
     ```

   - Replace rounded buttons with icon-only:

     ```jsx
     // V1
     <Button rounded><IconSymbol icon="add" /></Button>

     // V2
     <Button icon={{ icon: "add", position: "icon-only" }}>Add</Button>
     ```

```

When asked "How do I migrate Accordion component?":

```

To migrate the Accordion component from V1 to V2:

1. Import changes: Use V2 prefix with alias

   ```jsx
   // V1 import
   import { Accordion } from '@uhg-abyss/web/ui/Accordion';

   // V2 import
   import { V2Accordion as Accordion } from '@uhg-abyss/web/ui/Accordion';
   ```

2. Remove Accordion.Trigger:

   - V1: Uses nested Accordion.Trigger component
   - V2: Remove Trigger and move headingLevel to Header

3. Add triggerPosition prop:

   - Add triggerPosition="right" to main Accordion component

4. Update structure:
   - V1: `<Accordion.Trigger><Accordion.Header>Title</Accordion.Header></Accordion.Trigger>`
   - V2: `<Accordion.Header headingLevel={2}>Title</Accordion.Header>`

Would you like me to transform a specific code snippet for you?

````

## Code Transformation

When given a code snippet, transform it according to migration rules:

```jsx
// V1 code
import { Button } from '@uhg-abyss/web/ui/Button';
import { Accordion } from '@uhg-abyss/web/ui/Accordion';

<Button variant="solid" before={<IconSymbol icon="settings" />}>
  Settings
</Button>

<Accordion defaultValue="item-1" isCollapsible>
  <Accordion.Item value="item-1">
    <Accordion.Trigger headingLevel={2}>
      <Accordion.Header>Heading</Accordion.Header>
    </Accordion.Trigger>
    <Accordion.Content>Content</Accordion.Content>
  </Accordion.Item>
</Accordion>

// Transformed to V2
import { V2Button as Button } from '@uhg-abyss/web/ui/Button';
import { V2Accordion as Accordion } from '@uhg-abyss/web/ui/Accordion';

<Button variant="primary" icon={{ icon: "settings", position: "start" }}>
  Settings
</Button>

<Accordion defaultValue="item-1" isCollapsible triggerPosition="right">
  <Accordion.Item value="item-1">
    <Accordion.Header headingLevel={2}>Heading</Accordion.Header>
    <Accordion.Content>Content</Accordion.Content>
  </Accordion.Item>
</Accordion>
````
