# AI Migration Assistant Instructions

This file contains instructions for AI assistants to help migrate Abyss UI components from V1 to V2.

## Migration Context

The migration data is stored in the `MigrationData.json` file, which contains comprehensive migration information for all components. Each component entry in the JSON includes:

- `v1Import` and `v2Import` objects with import paths
- `overview` describing the key changes
- Component-specific change arrays (e.g., `buttonChanges`, `accordionChanges`)

## Assistance Tasks

When asked about migrating a component, the AI should:

1. Identify the component being migrated
2. Reference the appropriate migration data from `MigrationData.json`
3. Provide step-by-step migration instructions based on the component's change arrays
4. Offer to transform code snippets directly

## Component Migration Patterns

### Imports

- ALL V2 components must follow this import pattern: `import { V2ComponentName as ComponentName } from '@uhg-abyss/mobile/ui/ComponentName';`
- This applies to ALL components being migrated to V2, even if the component structure/name doesn't change
- Examples:
  - Button: `import { V2Button as Button } from '@uhg-abyss/mobile/ui/Button'`;
- This pattern preserves the component name in the JSX while using the V2 implementation

### Props

- Identify renamed props
- Find replacements for removed props
- Explain new prop patterns

### Component Structure

- Guide through structural changes in components
- Explain composition pattern changes

## Example Responses

When asked "How do I migrate Button component?":

````
To migrate the Button component from V1 to V2:

1. Import changes: Use V2 prefix with alias
   ```jsx
   // V1 import
   import { Button } from '@uhg-abyss/mobile/ui/Button';

   // V2 import
   import { V2Button as Button } from '@uhg-abyss/mobile/ui/Button';
   ```

2. Update variant values:

   - V1 variant values have been renamed and the default has changed:
     - 'primary' → 'brand'
     - 'secondary' → 'neutral' (now the default)
     - 'tertiary' → 'neutral' or style via tokens
     - 'destructive' → 'destructive'
     - 'alt' → 'inverse'

3. Update icon handling:

   - Replace before/after props with icon and iconPosition:

     ```jsx
     // V1
     <Button before={<IconSymbol icon="home" />}>Home</Button>
     <Button after={<IconSymbol icon="arrow_forward" />}>Next</Button>

     // V2
     <Button icon={{ name: "home" }} iconPosition="leading">Home</Button>
     <Button icon={{ name: "arrow_forward" }} iconPosition="trailing">Next</Button>
     ```

   - Replace rounded buttons with iconPosition="iconOnly":

     ```jsx
     // V1
     <Button rounded before={<IconSymbol icon="add" />} />

     // V2
     <Button icon={{ name: "add" }} iconPosition="iconOnly" />
     ```

````

When asked "How do I migrate Banner component?":

````
To migrate the Banner component from V1 to V2:

1. Import changes: Use V2 prefix with alias
   ```jsx
   // V1 import
   import { Banner } from '@uhg-abyss/mobile/ui/Banner';

   // V2 import
   import { V2Banner as Banner } from '@uhg-abyss/mobile/ui/Banner';
   ```

2. Update text prop names:

   - `title` → `heading`
   - `description` → `paragraph`

3. Update color/background prop:

   - `color` → `background`
   - Value changes:
     - '$white' → 'white'
     - '$pastel1' → 'aqua'
     - '$pastel2' → 'sky-blue'
     - '$pastel3' → 'peach'
     - '$pastel4' → 'mint'

4. Example transformation:

   ```jsx
   // V1
   <Banner
     title="Your Organizer"
     description={'A "one-stop-shop" for the important things a caregiver could need to give you for providing the best care.'}
     color="$white"
   />

   // V2
   <Banner
     heading="Your Organizer"
     paragraph={'A "one-stop-shop" for the important things a caregiver could need to give you for providing the best care.'}
     background="white"
   />
   ```

5. Update link/button handling (if needed):

   - Replace `linkText` and `href` with structured `cta` prop:

     ```jsx
     // V1
     <Banner linkText="Learn more" href="https://abyss.uhg.com/" />

     // V2
     <Banner cta={<Link href="https://abyss.uhg.com/">Learn more</Link>} />
     ```

6. Update variant values:
   - 'horizontal-large' → 'horizontal'
   - 'horizontal-small' → 'horizontal'
   - 'vertical' → 'vertical-lg' or 'vertical-sm'
   - 'branded' → 'branded'

````
